#=------------------------------------------------------------------------------
    Simulation of the solution of the nonlinear Schrödinger equation
    with white noise dispersion

    Details can be found in the article (see Section 5.2):
    [ ] A. Laurent, G. Vilmart
        Multirevolution integrators for differential equations
        with fast stochastic oscillations
        SIAM J. Sci. Comput. 42 (2020), no. 1, A115–A139.

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 9th October 2019

    Uses Julia 1.2.0
=#
#=------------------------------------------------------------------------------
     Parameters of the equation
=#
# Spatial length [-Lx,Lx]
Lx=pi

# FFT in space
N_FFT_x=64
K_FFT_x=2*N_FFT_x+1

# Polynomial nonlinearity
sigma=4
function F(u)
    return im*abs.(u).^(2*sigma).*u
end
function DF(u) # in 1D !
    if sigma==0
        return [0.0 -1.0 ; 1.0 0.0]
    else
        return [-2*sigma*real(u)*abs(u)^(2*sigma-2)*imag(u)  -2*sigma*imag(u)*abs(u)^(2*sigma-2)*imag(u)-abs(u)^(2*sigma) ; 2*sigma*real(u)*abs(u)^(2*sigma-2)*real(u)+abs(u)^(2*sigma)  2*sigma*imag(u)*abs(u)^(2*sigma-2)*real(u)]
    end
end

# Oscillation matrix
A=zeros(K_FFT_x,K_FFT_x)*im
for k=-N_FFT_x:N_FFT_x
    A[k+N_FFT_x+1,k+N_FFT_x+1]=-im*2*pi*k^2
end
#=------------------------------------------------------------------------------
    Numerical parameters of the problem
=#
N_FFT=32    # number of Fourier coefficients
K_FFT=2*N_FFT+1
epsilon=1.0e-2
N=10    # number of compositions
Mmax=150    # number of iterations of the algorithm
tol=1.0e-13     # fixed point tolerance for geometric method
K_fixed_point=100   # fixed point max iterations for geometric method
#=------------------------------------------------------------------------------
     Functions
=#
using Random
using DelimitedFiles
using LinearAlgebra
#Pkg.add("FFTW")
using FFTW

include("Functions.jl")
#=------------------------------------------------------------------------------
     Initial data
=#
function u_0(x)
    return exp.(-x.^2 .*(3*x.^2 .-1))
end

# Spatial Fourier coefficients of the initial condition u_0
xk=[k*2*Lx/K_FFT_x-2*Lx*(N_FFT_x)/K_FFT_x for k in 0:K_FFT_x-1]
u0shift=vcat(u_0(xk[N_FFT_x+1:end]),u_0(xk[1:N_FFT_x]))
aux=fft(u0shift)/K_FFT_x
u0_Fourier=zeros(K_FFT_x)*im
for k in 1:K_FFT_x
    if k<=N_FFT_x
        u0_Fourier[k]=aux[k+N_FFT_x+1]
    else
        u0_Fourier[k]=aux[k-N_FFT_x]
    end
end
#=------------------------------------------------------------------------------
    Random variables constants
=#
# Real covariance matrix of the alpha_k
L_alpha=zeros(2*K_FFT,2*K_FFT)
for p in -N_FFT:N_FFT
    P=p+N_FFT+1
    P_op=-p+N_FFT+1
    if p==0
        L_alpha[2*P-1:2*P,2*P-1:2*P]=[2/3 0.0; 0.0 0.0]
    else
        L_alpha[2*P-1:2*P,2*P-1:2*P]=[1/(2*pi^2*p^2) 0.0;0.0 1/(2*pi^2*p^2)]
        L_alpha[2*P-1:2*P,2*P_op-1:2*P_op]=[1/(2*pi^2*p^2) 0.0;0.0 -1/(2*pi^2*p^2)]
    end
end
Cov_alpha_aux=sqrt(L_alpha)

# Fixing the same randomness of the alpha_k for all algorithms
alpha_aux_aux=zeros(2*K_FFT,Mmax)
for M in 1:Mmax
    alpha_aux_aux[:,M]=2*bitrand(2*K_FFT) .-1
end
#=------------------------------------------------------------------------------
     Main script
=#
start = time()
println("Int1: ")
U_Fourier_1=Integrator_Fourier_weak_order_1(u0_Fourier,epsilon,N,Mmax)
println(time() - start)

start = time()
println("Int2: ")
U_Fourier_2=Integrator_Fourier_weak_order_2(u0_Fourier,epsilon,N,Mmax)
println(time() - start)

start = time()
println("Int1G: ")
U_Fourier_1G=Integrator_Fourier_weak_order_1_geometric(u0_Fourier,epsilon,N,Mmax,tol,K_fixed_point)
println(time() - start)

start = time()
println("Int2G: ")
U_Fourier_2G=Integrator_Fourier_weak_order_2_geometric(u0_Fourier,epsilon,N,Mmax,tol,K_fixed_point)
println(time() - start)
#=------------------------------------------------------------------------------
     Saving data
=#
List_parameters=[Lx,N_FFT_x,K_FFT_x,sigma,N_FFT,K_FFT,epsilon,N,Mmax,tol,K_fixed_point]
writedlm(string("./Data/List_parameters.txt"), List_parameters)
writedlm(string("./Data/u0_Fourier_re.txt"), real(u0_Fourier))
writedlm(string("./Data/u0_Fourier_im.txt"), imag(u0_Fourier))
writedlm(string("./Data/Int1_re.txt"), real(U_Fourier_1))
writedlm(string("./Data/Int1_im.txt"), imag(U_Fourier_1))
writedlm(string("./Data/Int2_re.txt"), real(U_Fourier_2))
writedlm(string("./Data/Int2_im.txt"), imag(U_Fourier_2))
writedlm(string("./Data/Int1G_re.txt"), real(U_Fourier_1G))
writedlm(string("./Data/Int1G_im.txt"), imag(U_Fourier_1G))
writedlm(string("./Data/Int2G_re.txt"), real(U_Fourier_2G))
writedlm(string("./Data/Int2G_im.txt"), imag(U_Fourier_2G))
