#=------------------------------------------------------------------------------
    Runge-Kutta integrators on the manifold M

    Details can be found in the article:
    [ ] A. Laurent, G. Vilmart
        Order conditions for sampling the invariant measure of ergodic
        stochastic differential equations on manifolds,
        Found. Comput. Math. 22, 649–695 (2022).

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 13th August 2020

    Uses Julia 1.4.2
=#
function Integrator_RK_manifold(A::Array{Float64,2},Alpha::Array{Float64,2},d::Array{Float64,1},X0::Array{Float64,1},h::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the constrained overdamped Langevin equation at time tN=h*N with the RK projection method given by the coefficients A, Alpha, and d.
    Input:
        A, Alpha, d Runge-Kutta coefficients, where A and Alpha are lower triangular matrices and A is strictly lower triangular
        X0 initial data
        h time stepsize
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        XN approximation of the solution at time tN=h*N with initial data X0
=#
    XN=X0
    s=size(A)[1]
    delta=Alpha*ones(s,1)
    fY_compute_bool=[!(norm(A[:,i])==0.) for i in 1:s]
    for n in 1:N
        Y=zeros(dim,s)
        fY=zeros(dim,s)
        gY=zeros(dim,s)
        U=rand(dim)
        xi=zeros(dim)
        for i in 1:dim
            if U[i]<1/6
                xi[i]=-sqrt(3)
            elseif U[i]>=5/6
                xi[i]=sqrt(3)
            end
        end
        for i in 1:s
            if delta[i]<0.5
                Y[:,i]=XN+sigma*sqrt(h)*d[i]*xi+h*fY*A[i,:]
            else
                incr=tol+1
                iter=0
                x=[XN;0]
                # Newton iteration
                while (norm(incr)>tol && iter<maxiter)
                    incr=[[Matrix{Float64}(I,dim,dim)-x[dim+1]*Alpha[i,i]*Dg(x[1:dim]) -(gY*Alpha[i,:]+Alpha[i,i]*g(x[1:dim]))] ;
                    [g(x[1:dim])' 0]]\[x[1:dim]-(XN+sigma*sqrt(h)*d[i]*xi+h*fY*A[i,:]+x[dim+1]*(gY*Alpha[i,:]+Alpha[i,i]*g(x[1:dim]))) ; zeta(x[1:dim])]
                    x=x-incr
                    iter=iter+1
                end
                Y[:,i]=x[1:dim]
            end
            if fY_compute_bool[i]
                fY[:,i]=f(Y[:,i])
            end
            gY[:,i]=g(Y[:,i])
        end
        XN=Y[:,s]
    end
    return XN
end


function Integrator_RK_manifold_sphere(A::Array{Float64,2},Alpha::Array{Float64,2},d::Array{Float64,1},X0::Array{Float64,1},h::Float64,N::Int64)
#=
    Author: Adrien Laurent
    Same algorithm as Integrator_RK_manifold in the particular case of the sphere (without using Newton iterations)
=#
    XN=X0
    s=size(A)[1]
    delta=Alpha*ones(s,1)
    fY_compute_bool=[!(norm(A[:,i])==0.) for i in 1:s]
    for n in 1:N
        Y=zeros(dim,s)
        fY=zeros(dim,s)
        U=rand(dim)
        xi=zeros(dim)
        for i in 1:dim
            if U[i]<1/6
                xi[i]=-sqrt(3)
            elseif U[i]>=5/6
                xi[i]=sqrt(3)
            end
        end
        for i in 1:s
            if delta[i]<0.5
                Y[:,i]=XN+sigma*sqrt(h)*d[i]*xi+h*fY*A[i,:]
            else
                auxA=XN+sigma*sqrt(h)*d[i]*xi+h*fY*A[i,:]
                auxB=Y*Alpha[i,:]
                normA2=auxA'*auxA
                normB2=auxB'*auxB
                scalAB=auxA'*auxB
                aux=Alpha[i,i]^2*normA2+2*Alpha[i,i]*scalAB+scalAB^2-normA2*normB2+normB2
                lambda=(Alpha[i,i]+scalAB-sign(Alpha[i,i]+scalAB)*sqrt(aux))/(Alpha[i,i]^2-normB2)
                Y[:,i]=1/(1-lambda*Alpha[i,i])*(auxA+lambda*auxB)
            end
            if fY_compute_bool[i]
                fY[:,i]=f(Y[:,i])
            end
        end
        XN=Y[:,s]
    end
    return XN
end


function Integrator_RK_manifold_SL(A::Array{Float64,2},Alpha::Array{Float64,2},d::Array{Float64,1},X0::Array{Float64,1},h::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Same algorithm as Integrator_RK_manifold, with the technical changes to work conveniently with matrices, and a rejection procedure if a substage is not invertible.
=#
    XN=X0
    s=size(A)[1]
    delta=Alpha*ones(s,1)
    fY_compute_bool=[!(norm(A[:,i])==0.) for i in 1:s]
    boolinv=1
    n=1
    while (n<=N && boolinv==1)
        Y=zeros(dim,s)
        fY=zeros(dim,s)
        gY=zeros(dim,s)
        U=rand(dim)
        xi=zeros(dim)
        for i in 1:dim
            if U[i]<1/6
                xi[i]=-sqrt(3)
            elseif U[i]>=5/6
                xi[i]=sqrt(3)
            end
        end
        for i in 1:s
            if delta[i]<0.5
                Y[:,i]=XN+sigma*sqrt(h)*d[i]*xi+h*fY*A[i,:]
            else
                incr=tol+1
                iter=0
                x=[XN;0]
                # Newton iteration
                while (norm(incr)>tol && iter<maxiter)
                    if det(reshape(x[1:dim],n_SL,n_SL))==0
                        boolinv=0
                    else
                        incr=[[Matrix{Float64}(I,dim,dim)-x[dim+1]*Alpha[i,i]*Dg(x[1:dim]) -(gY*Alpha[i,:]+Alpha[i,i]*g(x[1:dim]))] ;
                        [g(x[1:dim])' 0]]\[x[1:dim]-(XN+sigma*sqrt(h)*d[i]*xi+h*fY*A[i,:]+x[dim+1]*(gY*Alpha[i,:]+Alpha[i,i]*g(x[1:dim]))) ; zeta(x[1:dim])]
                        x=x-incr
                        iter=iter+1
                    end
                end
                if iter==maxiter
                    boolinv=0
                end
                Y[:,i]=x[1:dim]
            end
            if fY_compute_bool[i]
                fY[:,i]=f(Y[:,i])
            end
            if det(reshape(Y[:,i],n_SL,n_SL))==0
                boolinv=0
            else
                gY[:,i]=g(Y[:,i])
            end
        end
        XN=Y[:,s]
        n=n+1
    end
    return [XN,boolinv]
end
