#=------------------------------------------------------------------------------
    Invariant measure approximation on the special linear group

    Details can be found in the article (see Section 5.2):
    [ ] A. Laurent, G. Vilmart
        Order conditions for sampling the invariant measure of ergodic
        stochastic differential equations on manifolds,
        Found. Comput. Math. 22, 649–695 (2022).

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 13th August 2020

    Uses Julia 1.4.2
=#
#=------------------------------------------------------------------------------
    Manifold/Diffusion parameters
=#
using LinearAlgebra

n_SL=2
dim=n_SL^2  # Dimension of SL(n_SL)

zeta = x -> (X=reshape(x,n_SL,n_SL);det(X)-1)
g = x -> (X=reshape(x,n_SL,n_SL);det(X)*reshape(inv(X'),dim))   # g=nabla zeta
Dg = x -> (X=reshape(x,n_SL,n_SL);Y=inv(X);y=reshape(Y,dim);det(X)*(y*y'-kron(Y',Y)))

V0 = x -> 25*norm(x-reshape(Matrix{Float64}(I,n_SL,n_SL),dim))^2
f = x -> -50*(x-reshape(Matrix{Float64}(I,n_SL,n_SL),dim))  # f=-DV0

sigma=sqrt(2)

# Test function
phi = x -> tr(reshape(x,n_SL,n_SL))
#=------------------------------------------------------------------------------
    Numerical parameters of the problem
=#
T=10
N=2^14
maxiter=30
tol=1.0e-8
X0=reshape(Matrix{Float64}(I,n_SL,n_SL),dim)

N_trajectories=10
#=------------------------------------------------------------------------------
    Butcher Tableaux of the integrators
=#
# Euler with implicit projection direction
s_EI=2     # Xn+1=Ys
A_EI=[[0. 0.];[1. 0.]]
c_EI=[0.;1.]
Alpha_EI=[[0. 0.];[0. 1.]]
delta_EI=[0;1]
d_EI=[0.;1.]

# Order two Runge-Kutta integrator
s_2=4     # Xn+1=Ys
A_2=[[0. 0. 0. 0.];
    [0.621729189582953540 0. 0. 0.];
    [0 0.102032386582165330 0. 0.];
    [0.0547449506054026516 -0.0205123070437693053 0.9657673564383666538 0.]]
c_2=[0.;0.621729189582953540;0.102032386582165330;1.]
Alpha_2=[[1. 0. 0. 0.];
    [0.584372887990673524 0.415627112009326476 0. 0.];
    [0.887706593835748395 -0.345018694936693742 0.45731210110094535 0.];
    [0.0547449506054026516 -0.0205123070437693053 0.9657673564383666538 0.]]
delta_2=[1;1;1;1]
d_2=[-0.898931652839146019;-1.66233102561284629;0.318924515019668897;1.]
#=------------------------------------------------------------------------------
     Functions
=#
using Random
using DelimitedFiles

include("Functions.jl")
#=------------------------------------------------------------------------------
     Initialize random number generator
=#
if length(ARGS)==0
    Random.seed!(0)
else
    Random.seed!(parse(Int64,ARGS[1]))
end
#=------------------------------------------------------------------------------
     Main Script
=#
totaltime=time()
Anum=zeros(2*N_trajectories,2)
h=T/N

start = time()
for m in 1:N_trajectories
    println("m=",m)
    # aux_EI=Integrator_RK_manifold_SL(A_EI,Alpha_EI,d_EI,X0,h,N,tol,maxiter)
    aux_2=Integrator_RK_manifold_SL(A_2,Alpha_2,d_2,X0,h,N,tol,maxiter)
    # Anum[2*m-1,:]=[phi(aux_EI[1]),aux_EI[2]]
    Anum[2*m,:]=[phi(aux_2[1]),aux_2[2]]
end
println("total time = ",time()-totaltime)
#=------------------------------------------------------------------------------
     Storing results
=#
if length(ARGS)==0
    writedlm("./Data/data.txt", Anum)
else
    writedlm(string("./Data/res_",ARGS[1],".txt"), Anum)
end

println("end")
