#=------------------------------------------------------------------------------
    Invariant measure approximation on the sphere

    Details can be found in the article (see Section 5.1):
    [ ] A. Laurent, G. Vilmart
        Order conditions for sampling the invariant measure of ergodic
        stochastic differential equations on manifolds,
        Found. Comput. Math. 22, 649–695 (2022).

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 13th August 2020

    Uses Julia 1.4.2
=#
#=------------------------------------------------------------------------------
    Manifold/Diffusion parameters
=#
dim=3

zeta = x -> (x[1]^2+x[2]^2+x[3]^2-1)/2
g = x -> [x[1];x[2];x[3]]      # g=nabla zeta
Dg = x -> Matrix{Float64}(I,dim,dim)

Pi_M = x -> (Matrix{Float64}(I,dim,dim)- (g(x)'*g(x))^(-1)*g(x)*g(x)')

V0 = x -> 50*(1-x[1]^2-x[2]^2)/2
f = x -> -50*[-x[1] ; -x[2] ; 0]     # f=-DV0

sigma=sqrt(2)

# Test function
phi = x -> x[3]^2
#=------------------------------------------------------------------------------
    Numerical parameters of the problem
=#
T=20
liste_N = [floor(Int,2^(x)) for x in (5:1/2:10)]*T
lN=length(liste_N)
X0=[1.;0.;0.]

N_trajectories=10
#=------------------------------------------------------------------------------
    Butcher Tableaux
=#
# Euler with implicit projection direction
s_EI=2     # Xn+1=Ys
A_EI=[[0. 0.];[1. 0.]]
c_EI=[0.;1.]
Alpha_EI=[[0. 0.];[0. 1.]]
delta_EI=[0;1]
d_EI=[0.;1.]

# Order two Runge-Kutta integrator
s_2=4     # Xn+1=Ys
A_2=[[0. 0. 0. 0.];
    [0.621729189582953540 0. 0. 0.];
    [0 0.102032386582165330 0. 0.];
    [0.0547449506054026516 -0.0205123070437693053 0.9657673564383666538 0.]]
c_2=[0.;0.621729189582953540;0.102032386582165330;1.]
Alpha_2=[[1. 0. 0. 0.];
    [0.584372887990673524 0.415627112009326476 0. 0.];
    [0.887706593835748395 -0.345018694936693742 0.45731210110094535 0.];
    [0.0547449506054026516 -0.0205123070437693053 0.9657673564383666538 0.]]
delta_2=[1;1;1;1]
d_2=[-0.898931652839146019;-1.66233102561284629;0.318924515019668897;1.]
#=------------------------------------------------------------------------------
     Functions
=#
using Random
using DelimitedFiles
using LinearAlgebra

include("Functions.jl")
#=------------------------------------------------------------------------------
     Initialize random number generator
=#
if length(ARGS)==0
    Random.seed!(0)
else
    Random.seed!(parse(Int64,ARGS[1]))
end
#=------------------------------------------------------------------------------
     Main Script
=#
totaltime=time()
Snum_EI=zeros(lN)
Snum_2=zeros(lN)
for p in 1:lN
    N=liste_N[p]
    h=T/N
    println("p=",p)
    start = time()
    for m in 1:N_trajectories
        Snum_EI[p]+=phi(Integrator_RK_manifold_sphere(A_EI,Alpha_EI,d_EI,X0,h,N))
        Snum_2[p]+=phi(Integrator_RK_manifold_sphere(A_2,Alpha_2,d_2,X0,h,N))
    end
    println(time() - start)
end
Snum=[Snum_EI';Snum_2']/N_trajectories
println("total time = ",time()-totaltime)
#=------------------------------------------------------------------------------
     Storing results
=#
if length(ARGS)==0
    writedlm("./Data/data.txt", Snum)
else
    writedlm(string("./Data/res_",ARGS[1],".txt"), Snum)
end

println("end")
