#=------------------------------------------------------------------------------
    Auxiliary functions

    Details can be found in the article:
    [ ] A. Laurent
        A uniformly accurate scheme for the numerical integration of penalized Langevin dynamics
        To appear in SIAM J. Sci. Comput.

    PLEASE CITE THE ABOVE PAPER WHEN USING THIS PROGRAM ! :-)
    Version of 10.08.2022

    Uses Julia 1.5.4
=#
#=------------------------------------------------------------------------------
    Integrators in codimension q
=#
function Integrator_UA(X0::Array{Float64,1},h::Float64,epsi::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the penalized overdamped Langevin equation at time tN=h*N with the UA method.
    Input:
        X0 initial data
        h time stepsize
        epsi stiffness parameter
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        boo=false
        iter0=0
        fnum=f(Xnum)
        gnum=g(Xnum)
        invGnum=inv(G(Xnum))
        zetanum=zeta(Xnum)
        x=zeros(dim+codim)
        while (!(boo) && iter0<maxiter)
            U=rand(dim)
            xi=zeros(dim)
            for i in 1:dim
                if U[i]<1/6
                    xi[i]=-sqrt(3)
                elseif U[i]>=5/6
                    xi[i]=sqrt(3)
                end
            end
            incr=tol+1
            iter=0
            x[1:dim]=Xnum
            while (norm(incr)>tol && iter<maxiter)
                local corr_Fixman=zeros(dim)
                local corr=zeros(codim)
                local divgnum=zeros(codim)
                for i in 1:dim
                    e_i=Matrix{Float64}(I,dim,dim)[:,i]
                    corr_Fixman+=sigma^2/2*Dg(Xnum)(e_i)*invGnum*gnum'*e_i
                    corr+=(Dg(Xnum)(gnum*invGnum*gnum'*e_i-e_i))'*gnum*invGnum*gnum'*e_i
                    divgnum+=Dg(x)(e_i)'*e_i
                end
                incr=[[Matrix{Float64}(I,dim,dim) -gnum] ; [g(x[1:dim])' zeros(codim,codim)]]\
                    [x[1:dim]-(Xnum+sigma*sqrt(h)*xi+h*fnum+epsi/2*(1-exp(-2*h/epsi))*corr_Fixman+(1-exp(-h/epsi))^2/2*Dg(Xnum)(gnum*invGnum*zetanum)*invGnum*zetanum+gnum*x[dim+1:dim+codim]) ;
                    zeta(x[1:dim])-(exp(-h/epsi)*zetanum+sigma*(epsi/2*(1-exp(-2*h/epsi)))^(1/2)*gnum'*xi+epsi*(1-exp(-h/epsi))*(gnum'*fnum+gnum'*corr_Fixman+sigma^2/2*divgnum)+sigma^2*(epsi*(1-exp(-h/epsi))-(epsi*h/2*(1-exp(-2*h/epsi)))^(1/2))*corr)]
                x=x-incr
                iter=iter+1
            end
            if (norm(incr)<=tol && iter<maxiter)
                boo=true
            end
            iter0=iter0+1
        end
        Xnum=x[1:dim]
    end
    return Xnum
end


function Integrator_UA_matrix_form(X0::Array{Float64,1},h::Float64,epsi::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the penalized overdamped Langevin equation at time tN=h*N with the UA method.
    Input:
        X0 initial data
        h time stepsize
        epsi stiffness parameter
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        boo=false
        iter0=0
        fnum=f(Xnum)
        gnum=g(Xnum)
        invGnum=inv(G(Xnum))
        zetanum=zeta(Xnum)
        x=zeros(dim+codim)
        while (!(boo) && iter0<maxiter)
            U=rand(dim)
            xi=zeros(dim)
            for i in 1:dim
                if U[i]<1/6
                    xi[i]=-sqrt(3)
                elseif U[i]>=5/6
                    xi[i]=sqrt(3)
                end
            end
            incr=tol+1
            iter=0
            x[1:dim]=Xnum
            while (norm(incr)>tol && iter<maxiter)
                local corr_Fixman=zeros(dim)
                local corr=zeros(codim)
                for i in 1:dim
                    e_i=Matrix{Float64}(I,dim,dim)[:,i]
                    corr_Fixman+=sigma^2/2*Dg(Xnum)(e_i)*invGnum*gnum'*e_i
                    corr+=(Dg(Xnum)(gnum*invGnum*gnum'*e_i-e_i))'*gnum*invGnum*gnum'*e_i
                end
                incr=[[Matrix{Float64}(I,dim,dim) -gnum] ; [g(x[1:dim])' zeros(codim,codim)]]\
                    [x[1:dim]-(Xnum+sigma*sqrt(h)*xi+h*fnum+epsi/2*(1-exp(-2*h/epsi))*corr_Fixman+(1-exp(-h/epsi))^2/2*Dg(Xnum)(gnum*invGnum*zetanum)*invGnum*zetanum+gnum*x[dim+1:dim+codim]) ;
                    zeta(x[1:dim])-(exp(-h/epsi)*zetanum+sigma*(epsi/2*(1-exp(-2*h/epsi)))^(1/2)*gnum'*xi+epsi*(1-exp(-h/epsi))*(gnum'*fnum+gnum'*corr_Fixman+sigma^2/2*divg)+sigma^2*(epsi*(1-exp(-h/epsi))-(epsi*h/2*(1-exp(-2*h/epsi)))^(1/2))*corr)]
                x=x-incr
                iter=iter+1
            end
            if (norm(incr)<=tol && iter<maxiter)
                boo=true
            end
            iter0=iter0+1
        end
        Xnum=x[1:dim]
    end
    return Xnum
end


function Integrator_Explicit_Euler(X0::Array{Float64,1},h::Float64,epsi::Float64,N::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the penalized overdamped Langevin equation at time tN=h*N with explicit Euler.
    Input:
        X0 initial data
        h time stepsize
        epsi stiffness parameter
        N number of iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        gnum=g(Xnum)
        invGnum=inv(G(Xnum))
        U=rand(dim)
        xi=zeros(dim)
        for i in 1:dim
            if U[i]<1/6
                xi[i]=-sqrt(3)
            elseif U[i]>=5/6
                xi[i]=sqrt(3)
            end
        end
        corr_Fixman=zeros(dim)
        for i in 1:dim
            e_i=Matrix{Float64}(I,dim,dim)[:,i]
            corr_Fixman+=sigma^2/2*Dg(Xnum)(e_i)*invGnum*gnum'*e_i
        end
        Xnum=Xnum+sigma*sqrt(h)*xi+h*f(Xnum)+h*corr_Fixman-h/epsi*gnum*invGnum*zeta(Xnum)
    end
    return Xnum
end


function Integrator_Euler_Manifold(X0::Array{Float64,1},h::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the penalized overdamped Langevin equation at time tN=h*N with the constrained Euler method.
    Input:
        X0 initial data
        h time stepsize
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        boo=false
        iter0=0
        gnum=g(Xnum)
        x=zeros(dim+codim)
        while (!(boo) && iter0<maxiter)
            U=rand(dim)
            xi=zeros(dim)
            for i in 1:dim
                if U[i]<1/6
                    xi[i]=-sqrt(3)
                elseif U[i]>=5/6
                    xi[i]=sqrt(3)
                end
            end
            incr=tol+1
            iter=0
            x[1:dim]=Xnum
            while (norm(incr)>tol && iter<maxiter)
                incr=[[Matrix{Float64}(I,dim,dim) -gnum] ; [g(x[1:dim])' zeros(codim,codim)]]\
                    [x[1:dim]-(Xnum+sigma*sqrt(h)*xi+h*f(Xnum)+gnum*x[dim+1:dim+codim]) ;
                    zeta(x[1:dim])]
                x=x-incr
                iter=iter+1
            end
            if (norm(incr)<=tol && iter<maxiter)
                boo=true
            end
            iter0=iter0+1
        end
        Xnum=x[1:dim]
    end
    return Xnum
end
#=------------------------------------------------------------------------------
    Integrators - Codimension 1
=#
function Integrator_UA_codim_1(X0::Array{Float64,1},h::Float64,epsi::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the constrained overdamped Langevin equation at time tN=h*N with the UA method.
    Input:
        X0 initial data
        h time stepsize
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        boo=false
        x=zeros(dim+1)
        iter0=0
        while (!(boo) && iter0<maxiter)
            U=rand(dim)
            xi=zeros(dim)
            for i in 1:dim
                if U[i]<1/6
                    xi[i]=-sqrt(3)
                elseif U[i]>=5/6
                    xi[i]=sqrt(3)
                end
            end
            incr=tol+1
            iter=0
            x[1:dim]=Xnum
            x[dim+1]=0.
            while (norm(incr)>tol && iter<maxiter)
                incr=[[Matrix{Float64}(I,dim,dim) -g(Xnum)] ; [g(x[1:dim])' 0]]\
                    [x[1:dim]-(Xnum+sigma*sqrt(h)*xi+h*f(Xnum)+sigma^2*epsi/4*(1-exp(-2*h/epsi))*G(Xnum)^(-1)*Dg(Xnum)*g(Xnum)+(1-exp(-h/epsi))^2/2*zeta(Xnum)^2*G(Xnum)^(-2)*Dg(Xnum)*g(Xnum)+x[dim+1]*g(Xnum)) ;
                    zeta(x[1:dim])-(exp(-h/epsi)*zeta(Xnum)+sigma*(epsi/2*(1-exp(-2*h/epsi)))^(1/2)*g(Xnum)'*xi+epsi*(1-exp(-h/epsi))*(g(Xnum)'*f(Xnum)+sigma^2/2*G(Xnum)^(-1)*g(Xnum)'*Dg(Xnum)*g(Xnum)+sigma^2/2*divg(Xnum)))]
                x=x-incr
                iter=iter+1
            end
            if (norm(incr)<=tol && iter<maxiter)
                boo=true
            end
            iter0=iter0+1
        end
        Xnum=x[1:dim]
    end
    return Xnum
end


function Integrator_Explicit_Euler_codim_1(X0::Array{Float64,1},h::Float64,epsi::Float64,N::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the constrained overdamped Langevin equation at time tN=h*N with the explicit Euler method.
    Input:
        X0 initial data
        h time stepsize
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        U=rand(dim)
        xi=zeros(dim)
        for i in 1:dim
            if U[i]<1/6
                xi[i]=-sqrt(3)
            elseif U[i]>=5/6
                xi[i]=sqrt(3)
            end
        end
        Xnum=Xnum+sigma*sqrt(h)*xi+h*f(Xnum)+h*sigma^2/2*G(Xnum)^(-1)*Dg(Xnum)*g(Xnum)-h/epsi*g(Xnum)*G(Xnum)^(-1)*zeta(Xnum)
    end
    return Xnum
end


function Integrator_Euler_Manifold_codim_1(X0::Array{Float64,1},h::Float64,N::Int64,tol::Float64,maxiter::Int64)
#=
    Author: Adrien Laurent
    Computes an approximation of the solution of the constrained overdamped Langevin equation at time tN=h*N with the constrained Euler method.
    Input:
        X0 initial data
        h time stepsize
        N number of iterations
        tol, maxiter parameters for the Newton iterations
    Output:
        Xnum approximation of the solution at time tN=h*N with initial data X0
=#
    Xnum=X0
    for n in 1:N
        boo=false
        x=zeros(dim+1)
        iter0=0
        while (!(boo) && iter0<maxiter)
            U=rand(dim)
            xi=zeros(dim)
            for i in 1:dim
                if U[i]<1/6
                    xi[i]=-sqrt(3)
                elseif U[i]>=5/6
                    xi[i]=sqrt(3)
                end
            end
            incr=tol+1
            iter=0
            x[1:dim]=Xnum
            x[dim+1]=0.
            while (norm(incr)>tol && iter<maxiter)
                incr=[[Matrix{Float64}(I,dim,dim) -g(Xnum)] ; [g(x[1:dim])' 0]]\
                    [x[1:dim]-(Xnum+sigma*sqrt(h)*xi+h*f(Xnum)+x[dim+1]*g(Xnum)) ;
                    zeta(x[1:dim])]
                x=x-incr
                iter=iter+1
            end
            if (norm(incr)<=tol && iter<maxiter)
                boo=true
            end
            iter0=iter0+1
        end
        Xnum=x[1:dim]
    end
    return Xnum
end
